/*
 *  Copyright (C) 2004-2018 Savoir-faire Linux Inc.
 *
 *  Author: Aline Bonnet <aline.bonnet@savoirfairelinux.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package cx.ring.services;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;

import javax.inject.Inject;
import javax.inject.Named;

import cx.ring.daemon.Ringservice;
import cx.ring.daemon.StringVect;
import cx.ring.daemon.VectMap;
import cx.ring.model.CallContact;
import cx.ring.model.ServiceEvent;
import cx.ring.utils.FutureUtils;
import cx.ring.utils.Log;
import cx.ring.utils.Observable;

public class PresenceService extends Observable {
    private static final String TAG = PresenceService.class.getName();

    @Inject
    @Named("DaemonExecutor")
    ExecutorService mExecutor;

    @Inject
    DeviceRuntimeService mDeviceRuntimeService;

    Map<String, Boolean> mPresenceMap;

    public PresenceService() {
        mPresenceMap = new HashMap<>();
    }

    /**
     * Check service cache for latest presence value
     *
     * @param uri URI of the contact
     * @return true if this URI is online according to latest daemon update
     */
    public boolean isBuddyOnline(String uri) {
        if (uri != null && mPresenceMap.containsKey(uri)) {
            return mPresenceMap.get(uri);
        }
        return false;
    }

    public void publish(final String accountID, final boolean status, final String note) {
        FutureUtils.executeDaemonThreadCallable(
                mExecutor,
                mDeviceRuntimeService.provideDaemonThreadId(),
                false,
                () -> {
                    Ringservice.publish(accountID, status, note);
                    return true;
                }
        );
    }

    public void answerServerRequest(final String uri, final boolean flag) {
        FutureUtils.executeDaemonThreadCallable(
                mExecutor,
                mDeviceRuntimeService.provideDaemonThreadId(),
                false,
                () -> {
                    Ringservice.answerServerRequest(uri, flag);
                    return true;
                }
        );
    }

    public void subscribeBuddy(final String accountID, final String uri, final boolean flag) {
        FutureUtils.executeDaemonThreadCallable(
                mExecutor,
                mDeviceRuntimeService.provideDaemonThreadId(),
                false,
                () -> {
                    Ringservice.subscribeBuddy(accountID, uri, flag);
                    return true;
                }
        );
    }

    public VectMap getSubscriptions(final String accountID) {
        return FutureUtils.executeDaemonThreadCallable(
                mExecutor,
                mDeviceRuntimeService.provideDaemonThreadId(),
                true,
                () -> Ringservice.getSubscriptions(accountID)
        );
    }

    public void setSubscriptions(final String accountID, final StringVect uris) {
        FutureUtils.executeDaemonThreadCallable(
                mExecutor,
                mDeviceRuntimeService.provideDaemonThreadId(),
                false,
                () -> {
                    Ringservice.setSubscriptions(accountID, uris);
                    return true;
                }
        );
    }

    public void newServerSubscriptionRequest(String remote) {
        Log.d(TAG, "newServerSubscriptionRequest: " + remote);

        setChanged();
        ServiceEvent event = new ServiceEvent(ServiceEvent.EventType.NEW_SERVER_SUBSCRIPTION_REQUEST);
        event.addEventInput(ServiceEvent.EventInput.REMOTE, remote);
        notifyObservers(event);
    }

    public void serverError(String accountId, String error, String message) {
        Log.d(TAG, "serverError: " + accountId + ", " + error + ", " + message);

        setChanged();
        ServiceEvent event = new ServiceEvent(ServiceEvent.EventType.SERVER_ERROR);
        event.addEventInput(ServiceEvent.EventInput.ACCOUNT_ID, accountId);
        event.addEventInput(ServiceEvent.EventInput.ERROR, error);
        event.addEventInput(ServiceEvent.EventInput.MESSAGE, message);
        notifyObservers(event);
    }

    public void newBuddyNotification(String accountId, String buddyUri, int status, String lineStatus) {
        Log.d(TAG, "newBuddyNotification: " + accountId + ", " + buddyUri + ", " + status + ", " + lineStatus);

        mPresenceMap.put(CallContact.PREFIX_RING + buddyUri, status == 1);

        setChanged();
        ServiceEvent event = new ServiceEvent(ServiceEvent.EventType.NEW_BUDDY_NOTIFICATION);
        event.addEventInput(ServiceEvent.EventInput.ACCOUNT_ID, accountId);
        event.addEventInput(ServiceEvent.EventInput.BUDDY_URI, buddyUri);
        event.addEventInput(ServiceEvent.EventInput.STATE, status);
        event.addEventInput(ServiceEvent.EventInput.LINE_STATE, lineStatus);
        notifyObservers(event);
    }

    public void subscriptionStateChanged(String accountId, String buddyUri, int state) {
        Log.d(TAG, "subscriptionStateChanged: " + accountId + ", " + buddyUri + ", " + state);

        setChanged();
        ServiceEvent event = new ServiceEvent(ServiceEvent.EventType.SUBSCRIPTION_STATE_CHANGED);
        event.addEventInput(ServiceEvent.EventInput.ACCOUNT_ID, accountId);
        event.addEventInput(ServiceEvent.EventInput.BUDDY_URI, buddyUri);
        event.addEventInput(ServiceEvent.EventInput.STATE, state);
        notifyObservers(event);
    }
}

