/****************************************************************************
 *   Copyright (C) 2018 Savoir-faire Linux                                  *
 *   Author: Guillaume Roguez <guillaume.roguez@savoirfairelinux.com>       *
 *                                                                          *
 *   This library is free software; you can redistribute it and/or          *
 *   modify it under the terms of the GNU Lesser General Public             *
 *   License as published by the Free Software Foundation; either           *
 *   version 2.1 of the License, or (at your option) any later version.     *
 *                                                                          *
 *   This library is distributed in the hope that it will be useful,        *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *   Lesser General Public License for more details.                        *
 *                                                                          *
 *   You should have received a copy of the GNU General Public License      *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.  *
 ***************************************************************************/
#pragma once

// Std
#include <string>
#include <memory>
#include <ios>

// Qt
#include <qobject.h>

// Data
#include "api/datatransfer.h"
#include "api/account.h"

// LRC
#include "typedefs.h"

namespace lrc {

class CallbacksHandler;
class Database;

namespace api {

class BehaviorController;

/**
 *  @brief Class that manages data transfer.
 */
class LIB_EXPORT DataTransferModel : public QObject {
    Q_OBJECT

public:
    DataTransferModel(Database& database,
                      const CallbacksHandler& callbacksHandler,
                      const api::BehaviorController& behaviorController);
    ~DataTransferModel();

    std::vector<std::string> transferIdList() const;

    std::string sendFile(const std::string& account_id, const std::string& peer_uri,
                         const std::string& file_path, const std::string& display_name);

    datatransfer::Info transferInfo(const std::string& uid);

    std::streamsize bytesProgress(const std::string& id);

    void acceptFile(const std::string& id, const std::string& file_path, std::size_t offset);

    void cancel(const std::string& id);

Q_SIGNALS:
    /**
     * Connect this signal to know when a data transfer is incoming.
     * \note the unique identification is generated by the libring and its unicity scope is limited
     * to the libring process life.
     *
     * @param transfer_id unique identification of incoming data transfer.
     * @param display_name a free identification string given by sender.
     * @oaram size total number of bytes of the transfer (including offset).
     * @oaram offset offset of first given bytes for continued transfer.
     */
    void incomingTransfer(const std::string& uid, const std::string& display_name,
                          const std::size_t size, const std::size_t offset);

    /**
     * Connect this signal to know when an existing data transfer has changed of status.
     * @param transfer_id unique identification of incoming data transfer.
     * @param status reported status.
     */
    void transferStatusChanged(const std::string& uid, datatransfer::Status status);

private:
    class Impl;
    std::unique_ptr<Impl> pimpl_;
};

}} // namespace lrc::api
