/*
 * DCA compatible decoder - huffman tables
 * Copyright (C) 2004 Gildas Bazin
 * Copyright (C) 2007 Konstantin Shishkov
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "libavutil/common.h"

#include "avcodec.h"
#include "get_bits.h"
#include "dcahuff.h"

#define TMODE_COUNT 4
static const uint8_t tmode_vlc_bits[TMODE_COUNT] = { 3, 3, 3, 2 };
static const uint16_t tmode_codes[TMODE_COUNT][4] = {
    { 0x0000, 0x0002, 0x0006, 0x0007 },
    { 0x0002, 0x0006, 0x0007, 0x0000 },
    { 0x0006, 0x0007, 0x0000, 0x0002 },
    { 0x0000, 0x0001, 0x0002, 0x0003 }
};

static const uint8_t tmode_bits[TMODE_COUNT][4] = {
    { 1, 2, 3, 3 },
    { 2, 3, 3, 1 },
    { 3, 3, 1, 2 },
    { 2, 2, 2, 2 }
};

#define BITALLOC_12_COUNT    5
#define BITALLOC_12_VLC_BITS 9
static const uint8_t bitalloc_12_vlc_bits[BITALLOC_12_COUNT] = {
    9, 7, 7, 9, 9
};

static const uint16_t bitalloc_12_codes[BITALLOC_12_COUNT][12] = {
    { 0x0000, 0x0002, 0x0006, 0x000E, 0x001E, 0x003E, 0x00FF, 0x00FE,
      0x01FB, 0x01FA, 0x01F9, 0x01F8, },
    { 0x0001, 0x0000, 0x0002, 0x000F, 0x000C, 0x001D, 0x0039, 0x0038,
      0x0037, 0x0036, 0x0035, 0x0034, },
    { 0x0000, 0x0007, 0x0005, 0x0004, 0x0002, 0x000D, 0x000C, 0x0006,
      0x000F, 0x001D, 0x0039, 0x0038, },
    { 0x0003, 0x0002, 0x0000, 0x0002, 0x0006, 0x000E, 0x001E, 0x003E,
      0x007E, 0x00FE, 0x01FF, 0x01FE, },
    { 0x0001, 0x0000, 0x0002, 0x0006, 0x000E, 0x003F, 0x003D, 0x007C,
      0x0079, 0x0078, 0x00FB, 0x00FA, }
};

static const uint8_t bitalloc_12_bits[BITALLOC_12_COUNT][12] = {
    { 1, 2, 3, 4, 5, 6, 8, 8, 9, 9,  9,  9 },
    { 1, 2, 3, 5, 5, 6, 7, 7, 7, 7,  7,  7 },
    { 2, 3, 3, 3, 3, 4, 4, 4, 5, 6,  7,  7 },
    { 2, 2, 2, 3, 4, 5, 6, 7, 8, 9, 10, 10 },
    { 1, 2, 3, 4, 5, 7, 7, 8, 8, 8,  9,  9 }
};

#define SCALES_COUNT    5
#define SCALES_VLC_BITS 9
static const uint16_t scales_codes[SCALES_COUNT][129] = {
    { 0x3AB0, 0x3AB2, 0x3AB4, 0x3AB6, 0x3AB8, 0x3ABA, 0x3ABC, 0x3ABE,
      0x3AC0, 0x3AC2, 0x3AC4, 0x3AC6, 0x3AC8, 0x3ACA, 0x3ACC, 0x3ACE,
      0x3AD0, 0x3AD2, 0x3AD4, 0x3AD6, 0x3AD8, 0x3ADA, 0x3ADC, 0x3ADE,
      0x3AE0, 0x3AE2, 0x3AE4, 0x3AE6, 0x3AE8, 0x3AEA, 0x3AEC, 0x3AEE,
      0x3AF0, 0x3AF2, 0x3AF4, 0x3AF6, 0x3AF8, 0x3AFA, 0x3AFC, 0x3AFE,
      0x0540, 0x0542, 0x0544, 0x0546, 0x0548, 0x054A, 0x054C, 0x054E,
      0x0558, 0x055E, 0x02AD, 0x0154, 0x0754, 0x03A8, 0x0056, 0x0028,
      0x00E8, 0x004A, 0x000B, 0x003B, 0x0013, 0x0003, 0x000F, 0x0005,
      0x0001, 0x0006, 0x0000, 0x0008, 0x001C, 0x0004, 0x0024, 0x004B,
      0x00E9, 0x0029, 0x0057, 0x03A9, 0x0755, 0x0155, 0x02AE, 0x055F,
      0x0559, 0x054F, 0x054D, 0x054B, 0x0549, 0x0547, 0x0545, 0x0543,
      0x0541, 0x3AFF, 0x3AFD, 0x3AFB, 0x3AF9, 0x3AF7, 0x3AF5, 0x3AF3,
      0x3AF1, 0x3AEF, 0x3AED, 0x3AEB, 0x3AE9, 0x3AE7, 0x3AE5, 0x3AE3,
      0x3AE1, 0x3ADF, 0x3ADD, 0x3ADB, 0x3AD9, 0x3AD7, 0x3AD5, 0x3AD3,
      0x3AD1, 0x3ACF, 0x3ACD, 0x3ACB, 0x3AC9, 0x3AC7, 0x3AC5, 0x3AC3,
      0x3AC1, 0x3ABF, 0x3ABD, 0x3ABB, 0x3AB9, 0x3AB7, 0x3AB5, 0x3AB3,
      0x3AB1, },
    { 0x0F60, 0x0F62, 0x0F64, 0x0F66, 0x0F68, 0x0F6A, 0x0F6C, 0x0F6E,
      0x0F70, 0x0F72, 0x0F74, 0x0F76, 0x0F78, 0x0F7A, 0x0F7C, 0x0F7E,
      0x0F80, 0x0F82, 0x0F84, 0x0F86, 0x0F88, 0x0F8A, 0x0F8C, 0x0F8E,
      0x0F90, 0x0F92, 0x0F94, 0x0F96, 0x0F98, 0x0F9A, 0x0F9C, 0x0F9E,
      0x0FA0, 0x0FA2, 0x0FA4, 0x0FA6, 0x0FA8, 0x0FAA, 0x0FAC, 0x0FAE,
      0x0FB0, 0x0FB2, 0x0FB4, 0x0FB6, 0x0FB8, 0x0FBA, 0x0FBC, 0x0FBE,
      0x07A0, 0x07A2, 0x03D2, 0x01EA, 0x00FC, 0x007F, 0x001C, 0x000C,
      0x0004, 0x0034, 0x0010, 0x001B, 0x0009, 0x000B, 0x000E, 0x0001,
      0x0003, 0x0002, 0x000F, 0x000C, 0x000A, 0x0000, 0x0011, 0x0035,
      0x0005, 0x000D, 0x001D, 0x003C, 0x00FD, 0x01EB, 0x03D3, 0x07A3,
      0x07A1, 0x0FBF, 0x0FBD, 0x0FBB, 0x0FB9, 0x0FB7, 0x0FB5, 0x0FB3,
      0x0FB1, 0x0FAF, 0x0FAD, 0x0FAB, 0x0FA9, 0x0FA7, 0x0FA5, 0x0FA3,
      0x0FA1, 0x0F9F, 0x0F9D, 0x0F9B, 0x0F99, 0x0F97, 0x0F95, 0x0F93,
      0x0F91, 0x0F8F, 0x0F8D, 0x0F8B, 0x0F89, 0x0F87, 0x0F85, 0x0F83,
      0x0F81, 0x0F7F, 0x0F7D, 0x0F7B, 0x0F79, 0x0F77, 0x0F75, 0x0F73,
      0x0F71, 0x0F6F, 0x0F6D, 0x0F6B, 0x0F69, 0x0F67, 0x0F65, 0x0F63,
      0x0F61, },
    { 0x51D0, 0x51D2, 0x51D4, 0x51D6, 0x51D8, 0x51DA, 0x51DC, 0x51DE,
      0x51E0, 0x51E2, 0x51E4, 0x51E6, 0x51E8, 0x51EA, 0x51EC, 0x51EE,
      0x51F0, 0x51F2, 0x51F4, 0x51F6, 0x51F8, 0x51FA, 0x51FC, 0x51FE,
      0x70C0, 0x70C2, 0x70C4, 0x70C6, 0x70C8, 0x70CA, 0x70CC, 0x70CE,
      0x70EC, 0x10EA, 0x3868, 0x3877, 0x0876, 0x1C35, 0x0434, 0x0A34,
      0x0E1B, 0x021B, 0x051B, 0x070F, 0x010F, 0x0380, 0x0080, 0x0140,
      0x01C1, 0x0041, 0x00A1, 0x00E2, 0x0022, 0x0052, 0x0072, 0x0012,
      0x002A, 0x003A, 0x000A, 0x0016, 0x001E, 0x0006, 0x000C, 0x0000,
      0x0004, 0x0001, 0x000D, 0x0007, 0x001F, 0x0017, 0x000B, 0x003B,
      0x002B, 0x0013, 0x0073, 0x0053, 0x0023, 0x00E3, 0x00A2, 0x0042,
      0x01C2, 0x0141, 0x0081, 0x0381, 0x028C, 0x010C, 0x051C, 0x021C,
      0x0E1C, 0x0A35, 0x0435, 0x1C3A, 0x0877, 0x0874, 0x3869, 0x10EB,
      0x70ED, 0x70CF, 0x70CD, 0x70CB, 0x70C9, 0x70C7, 0x70C5, 0x70C3,
      0x70C1, 0x51FF, 0x51FD, 0x51FB, 0x51F9, 0x51F7, 0x51F5, 0x51F3,
      0x51F1, 0x51EF, 0x51ED, 0x51EB, 0x51E9, 0x51E7, 0x51E5, 0x51E3,
      0x51E1, 0x51DF, 0x51DD, 0x51DB, 0x51D9, 0x51D7, 0x51D5, 0x51D3,
      0x51D1, },
    { 0x6F64, 0x6F66, 0x6F68, 0x6F6A, 0x6F6C, 0x6F6E, 0x6F70, 0x6F72,
      0x6F74, 0x6F76, 0x6F78, 0x6F7A, 0x6F7C, 0x6F7E, 0x6F80, 0x6F82,
      0x6F84, 0x6F86, 0x6F88, 0x6F8A, 0x6F8C, 0x6F8E, 0x6F90, 0x6F92,
      0x6F94, 0x6F96, 0x6F98, 0x6F9A, 0x6F9C, 0x6F9E, 0x6FA0, 0x6FA2,
      0x6FA4, 0x6FA6, 0x6FA8, 0x6FAA, 0x6FAC, 0x6FAE, 0x6FB0, 0x6FB2,
      0x6FB4, 0x6FB6, 0x17B4, 0x37DC, 0x0BDB, 0x1BEF, 0x05EE, 0x0DF8,
      0x02F8, 0x06FD, 0x017D, 0x037F, 0x00BF, 0x0040, 0x00C0, 0x0021,
      0x0061, 0x0011, 0x0031, 0x0009, 0x0019, 0x0006, 0x000E, 0x0004,
      0x0000, 0x0005, 0x000F, 0x0007, 0x001A, 0x000A, 0x0036, 0x0016,
      0x006E, 0x002E, 0x00C1, 0x0041, 0x01BC, 0x00BC, 0x037A, 0x017A,
      0x02F9, 0x0DF9, 0x05EF, 0x05EC, 0x1BD8, 0x37DD, 0x17B5, 0x6FB7,
      0x6FB5, 0x6FB3, 0x6FB1, 0x6FAF, 0x6FAD, 0x6FAB, 0x6FA9, 0x6FA7,
      0x6FA5, 0x6FA3, 0x6FA1, 0x6F9F, 0x6F9D, 0x6F9B, 0x6F99, 0x6F97,
      0x6F95, 0x6F93, 0x6F91, 0x6F8F, 0x6F8D, 0x6F8B, 0x6F89, 0x6F87,
      0x6F85, 0x6F83, 0x6F81, 0x6F7F, 0x6F7D, 0x6F7B, 0x6F79, 0x6F77,
      0x6F75, 0x6F73, 0x6F71, 0x6F6F, 0x6F6D, 0x6F6B, 0x6F69, 0x6F67,
      0x6F65, },
    { 0xDF54, 0xDF56, 0xDFC8, 0xDFCA, 0xDFCC, 0xDFCE, 0xDFD0, 0xDFD2,
      0xDFD4, 0xDFD6, 0xDFD8, 0xDFDA, 0xDFDC, 0xDFDE, 0xDFE0, 0xDFE2,
      0x0FE8, 0x2FEA, 0x6FA8, 0x6FF6, 0x07F5, 0x07F7, 0x37D2, 0x37F9,
      0x03F8, 0x0BF8, 0x0BFB, 0x1BEB, 0x01FA, 0x05FA, 0x09FA, 0x0DFA,
      0x0DFF, 0x00FF, 0x02FF, 0x06FB, 0x007C, 0x017C, 0x027C, 0x027F,
      0x003C, 0x00BC, 0x013C, 0x01BC, 0x001C, 0x005C, 0x009C, 0x00DC,
      0x000C, 0x002C, 0x004C, 0x006C, 0x0004, 0x0014, 0x0024, 0x0034,
      0x0000, 0x0008, 0x0010, 0x0018, 0x001E, 0x0002, 0x0006, 0x000A,
      0x000E, 0x000B, 0x0007, 0x0003, 0x001F, 0x0019, 0x0011, 0x0009,
      0x0001, 0x0035, 0x0025, 0x0015, 0x0005, 0x006D, 0x004D, 0x002D,
      0x000D, 0x00DD, 0x009D, 0x005D, 0x001D, 0x01BD, 0x013D, 0x00BD,
      0x003D, 0x037C, 0x027D, 0x017D, 0x007D, 0x06FC, 0x04FC, 0x02FC,
      0x00FC, 0x0DFB, 0x09FB, 0x05FB, 0x01FB, 0x1BF8, 0x1BE8, 0x0BF9,
      0x03F9, 0x37FA, 0x37D3, 0x17F4, 0x07F6, 0x6FF7, 0x6FA9, 0x2FEB,
      0x0FE9, 0xDFE3, 0xDFE1, 0xDFDF, 0xDFDD, 0xDFDB, 0xDFD9, 0xDFD7,
      0xDFD5, 0xDFD3, 0xDFD1, 0xDFCF, 0xDFCD, 0xDFCB, 0xDFC9, 0xDF57,
      0xDF55, }
};

static const uint8_t scales_bits[SCALES_COUNT][129] = {
    { 14, 14, 14, 14, 14, 14, 14, 14,
      14, 14, 14, 14, 14, 14, 14, 14,
      14, 14, 14, 14, 14, 14, 14, 14,
      14, 14, 14, 14, 14, 14, 14, 14,
      14, 14, 14, 14, 14, 14, 14, 14,
      13, 13, 13, 13, 13, 13, 13, 13,
      13, 13, 12, 11, 11, 10,  9,  8,
       8,  7,  6,  6,  5,  4,  4,  3,
       2,  3,  3,  4,  5,  5,  6,  7,
       8,  8,  9, 10, 11, 11, 12, 13,
      13, 13, 13, 13, 13, 13, 13, 13,
      13, 14, 14, 14, 14, 14, 14, 14,
      14, 14, 14, 14, 14, 14, 14, 14,
      14, 14, 14, 14, 14, 14, 14, 14,
      14, 14, 14, 14, 14, 14, 14, 14,
      14, 14, 14, 14, 14, 14, 14, 14,
      14, },
    { 15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      14, 14, 13, 12, 11, 10,  8,  7,
       6,  6,  5,  5,  4,  4,  4,  3,
       3,  3,  4,  4,  4,  4,  5,  6,
       6,  7,  8,  9, 11, 12, 13, 14,
      14, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, },
    { 15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, 14, 14, 14, 13, 13, 12, 12,
      12, 11, 11, 11, 10, 10,  9,  9,
       9,  8,  8,  8,  7,  7,  7,  6,
       6,  6,  5,  5,  5,  4,  4,  3,
       3,  3,  4,  4,  5,  5,  5,  6,
       6,  6,  7,  7,  7,  8,  8,  8,
       9,  9,  9, 10, 10, 10, 11, 11,
      12, 12, 12, 13, 13, 13, 14, 14,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, },
    { 15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 14, 14, 13, 13, 12, 12,
      11, 11, 10, 10,  9,  8,  8,  7,
       7,  6,  6,  5,  5,  4,  4,  3,
       2,  3,  4,  4,  5,  5,  6,  6,
       7,  7,  8,  8,  9,  9, 10, 10,
      11, 12, 12, 12, 13, 14, 14, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, 15, 15, 15, 15, 15, 15, 15,
      15, },
    { 16, 16, 16, 16, 16, 16, 16, 16,
      16, 16, 16, 16, 16, 16, 16, 16,
      15, 15, 15, 15, 14, 14, 14, 14,
      13, 13, 13, 13, 12, 12, 12, 12,
      12, 11, 11, 11, 10, 10, 10, 10,
       9,  9,  9,  9,  8,  8,  8,  8,
       7,  7,  7,  7,  6,  6,  6,  6,
       5,  5,  5,  5,  5,  4,  4,  4,
       4,  4,  4,  4,  5,  5,  5,  5,
       5,  6,  6,  6,  6,  7,  7,  7,
       7,  8,  8,  8,  8,  9,  9,  9,
       9, 10, 10, 10, 10, 11, 11, 11,
      11, 12, 12, 12, 12, 13, 13, 13,
      13, 14, 14, 14, 14, 15, 15, 15,
      15, 16, 16, 16, 16, 16, 16, 16,
      16, 16, 16, 16, 16, 16, 16, 16,
      16,
    }
};

static const uint16_t bitalloc_3_codes[3] = {
    0x0003, 0x0000, 0x0002,
};

static const uint8_t bitalloc_3_bits[3] = {
    2,  1,  2,
};

static const uint16_t bitalloc_5_codes_a[5] = {
    0x000F, 0x0006, 0x0000, 0x0002, 0x000E,
};

static const uint16_t bitalloc_5_codes_b[5] = {
    0x0007, 0x0001, 0x0002, 0x0000, 0x0006,
};

static const uint16_t bitalloc_5_codes_c[5] = {
    0x0007, 0x0005, 0x0000, 0x0004, 0x0006,
};

static const uint8_t bitalloc_5_bits_a[5] = {
    4,  3,  1,  2,  4,
};

static const uint8_t bitalloc_5_bits_b[5] = {
    3,  2,  2,  2,  3,
};

static const uint8_t bitalloc_5_bits_c[5] = {
    3,  3,  1,  3,  3,
};

static const uint16_t bitalloc_7_codes_a[7] = {
    0x001E, 0x000E, 0x0005, 0x0000, 0x0006, 0x0004, 0x001F,
};

static const uint16_t bitalloc_7_codes_b[7] = {
    0x0014, 0x000B, 0x0000, 0x0003, 0x0001, 0x0004, 0x0015,
};

static const uint16_t bitalloc_7_codes_c[7] = {
    0x0000, 0x0002, 0x0001, 0x0003, 0x0002, 0x0003, 0x0001,
};

static const uint8_t bitalloc_7_bits_a[7] = {
    5,  4,  3,  1,  3,  3,  5,
};

static const uint8_t bitalloc_7_bits_b[7] = {
    5,  4,  2,  2,  2,  3,  5,
};

static const uint8_t bitalloc_7_bits_c[7] = {
    4,  4,  2,  2,  2,  4,  4,
};

static const uint16_t bitalloc_9_codes_a[9] = {
    0x0030, 0x0019, 0x0009, 0x0005, 0x0000, 0x0007, 0x000D, 0x0008,
    0x0031,
};

static const uint16_t bitalloc_9_codes_b[9] = {
    0x0018, 0x001A, 0x0002, 0x0007, 0x0002, 0x0000, 0x0003, 0x001B,
    0x0019,
};

static const uint16_t bitalloc_9_codes_c[9] = {
    0x001C, 0x000F, 0x0002, 0x0007, 0x0002, 0x0000, 0x0006, 0x0006,
    0x001D,
};

static const uint8_t bitalloc_9_bits_a[9] = {
    6,  5,  4,  3,  1,  3,  4,  4,  6,
};

static const uint8_t bitalloc_9_bits_b[9] = {
    5,  5,  3,  3,  2,  2,  3,  5,  5,
};

static const uint8_t bitalloc_9_bits_c[9] = {
    6,  5,  3,  3,  2,  2,  3,  4,  6,
};

static const uint16_t bitalloc_13_codes_a[13] = {
    0x0070, 0x002E, 0x0039, 0x001D, 0x000C, 0x000F, 0x0000, 0x0004,
    0x000D, 0x000A, 0x0016, 0x002F, 0x0071,
};

static const uint16_t bitalloc_13_codes_b[13] = {
    0x0038, 0x0010, 0x001D, 0x0007, 0x000F, 0x0005, 0x0000, 0x0006,
    0x0002, 0x0009, 0x0006, 0x0011, 0x0039,
};

static const uint16_t bitalloc_13_codes_c[13] = {
    0x0004, 0x001A, 0x0003, 0x000E, 0x0000, 0x0003, 0x0005, 0x0004,
    0x0002, 0x000F, 0x000C, 0x001B, 0x0005,
};

static const uint8_t bitalloc_13_bits_a[13] = {
     7,  6,  6,  5,  4,  4,  1,  3,  4,  4,  5,  6,  7,
};

static const uint8_t bitalloc_13_bits_b[13] = {
     6,  5,  5,  4,  4,  3,  2,  3,  3,  4,  4,  5,  6,
};

static const uint8_t bitalloc_13_bits_c[13] = {
     5,  5,  4,  4,  3,  3,  3,  3,  3,  4,  4,  5,  5,
};

static const uint16_t bitalloc_17_codes_a[17] = {
    0x0154, 0x00AB, 0x002B, 0x000B, 0x0003, 0x000A, 0x0001, 0x0006,
    0x0001, 0x0007, 0x0004, 0x000B, 0x0000, 0x0004, 0x0014, 0x0054,
    0x0155,
};

static const uint16_t bitalloc_17_codes_b[17] = {
    0x007C, 0x003F, 0x0019, 0x000D, 0x001C, 0x0008, 0x000F, 0x0005,
    0x0000, 0x0006, 0x0002, 0x0009, 0x001D, 0x000E, 0x001E, 0x0018,
    0x007D,
};

static const uint16_t bitalloc_17_codes_c[17] = {
    0x002C, 0x0017, 0x0005, 0x001C, 0x0003, 0x000A, 0x000F, 0x0003,
    0x0006, 0x0004, 0x0000, 0x000B, 0x0004, 0x001D, 0x000A, 0x0004,
    0x002D,
};

static const uint16_t bitalloc_17_codes_d[17] = {
    0x0100, 0x0102, 0x0082, 0x0042, 0x0022, 0x0012, 0x000A, 0x0006,
    0x0000, 0x0007, 0x000B, 0x0013, 0x0023, 0x0043, 0x0083, 0x0103,
    0x0101,
};

static const uint16_t bitalloc_17_codes_e[17] = {
    0x00E8, 0x00F6, 0x0075, 0x0034, 0x003B, 0x001B, 0x001F, 0x0004,
    0x0000, 0x0005, 0x000C, 0x001C, 0x003C, 0x0035, 0x007A, 0x00F7,
    0x00E9,
};

static const uint16_t bitalloc_17_codes_f[17] = {
    0x0004, 0x0003, 0x001E, 0x0001, 0x0001, 0x000E, 0x0001, 0x0004,
    0x0006, 0x0005, 0x0002, 0x000F, 0x0006, 0x000E, 0x001F, 0x0000,
    0x0005,
};

static const uint16_t bitalloc_17_codes_g[17] = {
    0x0060, 0x007E, 0x0031, 0x0019, 0x000D, 0x0004, 0x0000, 0x0006,
    0x0002, 0x0007, 0x0001, 0x0005, 0x000E, 0x001E, 0x003E, 0x007F,
    0x0061,
};

static const uint8_t bitalloc_17_bits_a[17] = {
    12, 11,  9,  7,  5,  4,  3,  3,  2,  3,  3,  4,  4,  6,  8, 10,
    12,
};

static const uint8_t bitalloc_17_bits_b[17] = {
    8,  7,  6,  5,  5,  4,  4,  3,  2,  3,  3,  4,  5,  5,  6,  6,
    8,
};

static const uint8_t bitalloc_17_bits_c[17] = {
    7,  6,  5,  5,  4,  4,  4,  3,  3,  3,  3,  4,  4,  5,  5,  5,
    7,
};

static const uint8_t bitalloc_17_bits_d[17] = {
    9,  9,  8,  7,  6,  5,  4,  3,  1,  3,  4,  5,  6,  7,  8,  9,
    9,
};

static const uint8_t bitalloc_17_bits_e[17] = {
    8,  8,  7,  6,  6,  5,  5,  3,  1,  3,  4,  5,  6,  6,  7,  8,
    8,
};

static const uint8_t bitalloc_17_bits_f[17] = {
    8,  7,  6,  5,  4,  4,  3,  3,  3,  3,  3,  4,  4,  5,  6,  6,
    8,
};

static const uint8_t bitalloc_17_bits_g[17] = {
    8,  8,  7,  6,  5,  4,  3,  3,  2,  3,  3,  4,  5,  6,  7,  8,
    8,
};

static const uint16_t bitalloc_25_codes_a[25] = {
    0x2854, 0x142B, 0x050B, 0x0143, 0x00A2, 0x0052, 0x002E, 0x0015,
    0x0004, 0x000E, 0x0000, 0x0003, 0x0006, 0x0004, 0x0001, 0x000F,
    0x0005, 0x0016, 0x002F, 0x0053, 0x00A3, 0x00A0, 0x0284, 0x0A14,
    0x2855,
};

static const uint16_t bitalloc_25_codes_b[25] = {
    0x001C, 0x000F, 0x0005, 0x0000, 0x0030, 0x0036, 0x000E, 0x0019,
    0x0001, 0x0008, 0x000E, 0x0001, 0x0005, 0x0002, 0x000F, 0x0009,
    0x0006, 0x001A, 0x000F, 0x0037, 0x0031, 0x0001, 0x0006, 0x0004,
    0x001D,
};

static const uint16_t bitalloc_25_codes_c[25] = {
    0x004C, 0x0027, 0x006D, 0x0028, 0x0037, 0x000E, 0x0015, 0x0000,
    0x0005, 0x0008, 0x000B, 0x000E, 0x0001, 0x000F, 0x000C, 0x0009,
    0x0006, 0x0001, 0x001A, 0x000F, 0x0008, 0x0029, 0x0012, 0x006C,
    0x004D,
};

static const uint16_t bitalloc_25_codes_d[25] = {
    0x0780, 0x0782, 0x03C2, 0x01E2, 0x00FE, 0x0079, 0x003D, 0x001C,
    0x000C, 0x0004, 0x0000, 0x0006, 0x0002, 0x0007, 0x0001, 0x0005,
    0x000D, 0x001D, 0x003E, 0x007E, 0x00FF, 0x01E3, 0x03C3, 0x0783,
    0x0781,
};

static const uint16_t bitalloc_25_codes_e[25] = {
    0x003C, 0x0092, 0x0018, 0x001F, 0x004E, 0x000D, 0x0025, 0x0004,
    0x0010, 0x0000, 0x000A, 0x0002, 0x0003, 0x0003, 0x000B, 0x0001,
    0x0011, 0x0005, 0x0026, 0x000E, 0x004F, 0x0048, 0x0019, 0x0093,
    0x003D,
};

static const uint16_t bitalloc_25_codes_f[25] = {
    0x0324, 0x0193, 0x00CE, 0x0065, 0x0024, 0x000C, 0x0013, 0x0004,
    0x0007, 0x000A, 0x000D, 0x000F, 0x0001, 0x0000, 0x000E, 0x000B,
    0x0008, 0x0005, 0x0018, 0x000D, 0x0025, 0x0066, 0x00CF, 0x00C8,
    0x0325,
};

static const uint16_t bitalloc_25_codes_g[25] = {
    0x03A8, 0x03AE, 0x01D5, 0x0094, 0x0014, 0x004B, 0x000B, 0x003B,
    0x0013, 0x0003, 0x000F, 0x0005, 0x0001, 0x0006, 0x0000, 0x0008,
    0x001C, 0x0004, 0x0024, 0x0074, 0x0015, 0x0095, 0x01D6, 0x03AF,
    0x03A9,
};

static const uint8_t bitalloc_25_bits_a[25] = {
    14, 13, 11,  9,  8,  7,  6,  5,  4,  4,  3,  3,  3,  3,  3,  4,
     4,  5,  6,  7,  8,  8, 10, 12, 14,
};

static const uint8_t bitalloc_25_bits_b[25] = {
    9,  8,  7,  6,  6,  6,  5,  5,  4,  4,  4,  3,  3,  3,  4,  4,
    4,  5,  5,  6,  6,  6,  7,  7,  9,
};

static const uint8_t bitalloc_25_bits_c[25] = {
    8,  7,  7,  6,  6,  5,  5,  4,  4,  4,  4,  4,  3,  4,  4,  4,
    4,  4,  5,  5,  5,  6,  6,  7,  8,
};

static const uint8_t bitalloc_25_bits_d[25] = {
    12, 12, 11, 10,  9,  8,  7,  6,  5,  4,  3,  3,  2,  3,  3,  4,
     5,  6,  7,  8,  9, 10, 11, 12, 12,
};

static const uint8_t bitalloc_25_bits_e[25] = {
    8,  8,  7,  7,  7,  6,  6,  5,  5,  4,  4,  3,  2,  3,  4,  4,
    5,  5,  6,  6,  7,  7,  7,  8,  8,
};

static const uint8_t bitalloc_25_bits_f[25] = {
    10,  9,  8,  7,  6,  5,  5,  4,  4,  4,  4,  4,  3,  3,  4,  4,
     4,  4,  5,  5,  6,  7,  8,  8, 10,
};

static const uint8_t bitalloc_25_bits_g[25] = {
    10, 10,  9,  8,  7,  7,  6,  6,  5,  4,  4,  3,  2,  3,  3,  4,
     5,  5,  6,  7,  7,  8,  9, 10, 10,
};

static const uint16_t bitalloc_33_codes_a[33] = {
    0x1580, 0x1582, 0x0AC2, 0x0562, 0x02B2, 0x015E, 0x00AD, 0x0054,
    0x001C, 0x003C, 0x000F, 0x001F, 0x0008, 0x000B, 0x000D, 0x0000,
    0x0002, 0x0001, 0x000E, 0x000C, 0x0009, 0x0006, 0x0014, 0x003D,
    0x001D, 0x0055, 0x00AE, 0x015F, 0x02B3, 0x0563, 0x0AC3, 0x1583,
    0x1581,
};

static const uint16_t bitalloc_33_codes_b[33] = {
    0x030C, 0x0187, 0x006D, 0x0028, 0x0037, 0x0066, 0x0015, 0x0031,
    0x0000, 0x000B, 0x0012, 0x001A, 0x0001, 0x0007, 0x000A, 0x000E,
    0x0001, 0x000F, 0x000B, 0x0008, 0x0004, 0x001B, 0x0013, 0x000C,
    0x0001, 0x0032, 0x001A, 0x0067, 0x0060, 0x0029, 0x00C2, 0x006C,
    0x030D,
};

static const uint16_t bitalloc_33_codes_c[33] = {
    0x00CC, 0x0067, 0x0005, 0x0070, 0x0003, 0x001A, 0x0039, 0x003F,
    0x000A, 0x0012, 0x0018, 0x001D, 0x0001, 0x0003, 0x0007, 0x000A,
    0x000D, 0x000B, 0x0008, 0x0004, 0x0002, 0x001E, 0x0019, 0x0013,
    0x000B, 0x0000, 0x003E, 0x001B, 0x0018, 0x0071, 0x0032, 0x0004,
    0x00CD,
};

static const uint16_t bitalloc_33_codes_d[33] = {
    0x3AF8, 0x3AFA, 0x1D7E, 0x0EBC, 0x075C, 0x03AC, 0x01D4, 0x0094,
    0x0014, 0x004B, 0x000B, 0x003B, 0x0013, 0x0003, 0x000F, 0x0005,
    0x0001, 0x0006, 0x0000, 0x0008, 0x001C, 0x0004, 0x0024, 0x0074,
    0x0015, 0x0095, 0x01D5, 0x03AD, 0x075D, 0x0EBD, 0x1D7F, 0x3AFB,
    0x3AF9,
};

static const uint16_t bitalloc_33_codes_e[33] = {
    0x01C8, 0x01E6, 0x0064, 0x00E2, 0x00E5, 0x0030, 0x0033, 0x0073,
    0x007A, 0x001A, 0x003A, 0x0002, 0x001A, 0x001F, 0x0007, 0x0001,
    0x0002, 0x0002, 0x000C, 0x0000, 0x001B, 0x0003, 0x003B, 0x001B,
    0x007B, 0x0078, 0x0070, 0x0031, 0x00F2, 0x00E3, 0x0065, 0x01E7,
    0x01C9,
};

static const uint16_t bitalloc_33_codes_f[33] = {
    0x0724, 0x0393, 0x01CE, 0x00E5, 0x002C, 0x0008, 0x0017, 0x003E,
    0x0005, 0x0014, 0x001D, 0x0000, 0x0003, 0x0006, 0x0008, 0x000B,
    0x000D, 0x000C, 0x0009, 0x0007, 0x0004, 0x0001, 0x001E, 0x0015,
    0x000A, 0x003F, 0x0038, 0x0009, 0x002D, 0x00E6, 0x01CF, 0x01C8,
    0x0725,
};

static const uint16_t bitalloc_33_codes_g[33] = {
    0x0284, 0x0042, 0x0140, 0x0143, 0x003E, 0x00BE, 0x0011, 0x0051,
    0x0009, 0x0029, 0x0005, 0x0015, 0x0000, 0x0008, 0x000E, 0x0002,
    0x0006, 0x0003, 0x000F, 0x0009, 0x0001, 0x0016, 0x0006, 0x002E,
    0x000E, 0x005E, 0x001E, 0x00BF, 0x003F, 0x0020, 0x0141, 0x0043,
    0x0285,
};

static const uint8_t bitalloc_33_bits_a[33] = {
    13, 13, 12, 11, 10,  9,  8,  7,  6,  6,  5,  5,  4,  4,  4,  3,
     3,  3,  4,  4,  4,  4,  5,  6,  6,  7,  8,  9, 10, 11, 12, 13,
    13,
};

static const uint8_t bitalloc_33_bits_b[33] = {
    10,  9,  8,  7,  7,  7,  6,  6,  5,  5,  5,  5,  4,  4,  4,  4,
     3,  4,  4,  4,  4,  5,  5,  5,  5,  6,  6,  7,  7,  7,  8,  8,
    10,
};

static const uint8_t bitalloc_33_bits_c[33] = {
    9,  8,  7,  7,  6,  6,  6,  6,  5,  5,  5,  5,  4,  4,  4,  4,
    4,  4,  4,  4,  4,  5,  5,  5,  5,  5,  6,  6,  6,  7,  7,  7,
    9,
};

static const uint8_t bitalloc_33_bits_d[33] = {
    14, 14, 13, 12, 11, 10,  9,  8,  7,  7,  6,  6,  5,  4,  4,  3,
     2,  3,  3,  4,  5,  5,  6,  7,  7,  8,  9, 10, 11, 12, 13, 14,
    14,
};

static const uint8_t bitalloc_33_bits_e[33] = {
    9,  9,  8,  8,  8,  7,  7,  7,  7,  6,  6,  5,  5,  5,  4,  3,
    2,  3,  4,  4,  5,  5,  6,  6,  7,  7,  7,  7,  8,  8,  8,  9,
    9,
};

static const uint8_t bitalloc_33_bits_f[33] = {
    11, 10,  9,  8,  7,  6,  6,  6,  5,  5,  5,  4,  4,  4,  4,  4,
     4,  4,  4,  4,  4,  4,  5,  5,  5,  6,  6,  6,  7,  8,  9,  9,
    11,
};

static const uint8_t bitalloc_33_bits_g[33] = {
    10,  9,  9,  9,  8,  8,  7,  7,  6,  6,  5,  5,  4,  4,  4,  3,
     3,  3,  4,  4,  4,  5,  5,  6,  6,  7,  7,  8,  8,  8,  9,  9,
    10,
};

static const uint16_t bitalloc_65_codes_a[65] = {
    0x9E5C, 0x9E5E, 0x4F2C, 0x2794, 0x13C4, 0x1E44, 0x09E3, 0x0F23,
    0x04F3, 0x0792, 0x027E, 0x03CE, 0x013D, 0x01E5, 0x009C, 0x00CC,
    0x0040, 0x0058, 0x0067, 0x001E, 0x0021, 0x002D, 0x003D, 0x0007,
    0x0011, 0x0014, 0x0017, 0x001A, 0x001C, 0x001F, 0x0001, 0x0004,
    0x0006, 0x0005, 0x0002, 0x0000, 0x001D, 0x001B, 0x0018, 0x0015,
    0x0012, 0x000E, 0x0006, 0x0032, 0x0026, 0x001F, 0x0078, 0x0059,
    0x0041, 0x00CD, 0x009D, 0x01E6, 0x013E, 0x03CF, 0x027F, 0x0793,
    0x0790, 0x04F0, 0x09E4, 0x1E45, 0x13C5, 0x2795, 0x4F2D, 0x9E5F,
    0x9E5D,
};

static const uint16_t bitalloc_65_codes_b[65] = {
    0x0A8C, 0x0547, 0x01B5, 0x0008, 0x00DB, 0x0152, 0x0005, 0x000B,
    0x008E, 0x00AE, 0x00E4, 0x0003, 0x0037, 0x0039, 0x0055, 0x006C,
    0x0073, 0x0003, 0x0015, 0x001D, 0x0028, 0x0030, 0x0037, 0x003E,
    0x0006, 0x000B, 0x000F, 0x0012, 0x0016, 0x0019, 0x001D, 0x0001,
    0x0004, 0x0002, 0x001E, 0x001A, 0x0017, 0x0013, 0x0010, 0x000C,
    0x0007, 0x003F, 0x0038, 0x0031, 0x0029, 0x0022, 0x001A, 0x0014,
    0x0000, 0x006D, 0x0056, 0x0046, 0x0038, 0x0004, 0x00E5, 0x00AF,
    0x008F, 0x006C, 0x000A, 0x0153, 0x0150, 0x0009, 0x02A2, 0x01B4,
    0x0A8D,
};

static const uint16_t bitalloc_65_codes_c[65] = {
    0x045C, 0x022F, 0x03F5, 0x01BC, 0x01FB, 0x0059, 0x00D0, 0x00DF,
    0x000A, 0x002D, 0x002F, 0x0052, 0x0069, 0x0078, 0x007F, 0x000A,
    0x0010, 0x001C, 0x0023, 0x002A, 0x0035, 0x003A, 0x003D, 0x0000,
    0x0003, 0x0006, 0x0009, 0x000C, 0x000F, 0x0012, 0x0016, 0x0018,
    0x001C, 0x0019, 0x0017, 0x0013, 0x0010, 0x000D, 0x000A, 0x0007,
    0x0004, 0x0001, 0x003E, 0x003B, 0x0036, 0x002B, 0x0028, 0x001D,
    0x0011, 0x000B, 0x0004, 0x0079, 0x006E, 0x0053, 0x0044, 0x002E,
    0x000B, 0x00FC, 0x00D1, 0x008A, 0x0058, 0x01BD, 0x0116, 0x03F4,
    0x045D,
};

static const uint16_t bitalloc_65_codes_d[65] = {
    0x70B0, 0x70B2, 0x70B4, 0x2852, 0x385B, 0x142E, 0x1C2E, 0x0A15,
    0x0E14, 0x0214, 0x0704, 0x0104, 0x010B, 0x0383, 0x0083, 0x0143,
    0x01C3, 0x0043, 0x00A2, 0x00E2, 0x0022, 0x0052, 0x0072, 0x0012,
    0x002A, 0x003A, 0x000A, 0x0016, 0x001E, 0x0006, 0x000C, 0x0000,
    0x0004, 0x0001, 0x000D, 0x0007, 0x001F, 0x0017, 0x000B, 0x003B,
    0x002B, 0x0013, 0x0073, 0x0053, 0x0023, 0x00E3, 0x00A3, 0x00A0,
    0x0040, 0x01C0, 0x0084, 0x0384, 0x0284, 0x0105, 0x0705, 0x0215,
    0x0E15, 0x0A16, 0x1C2F, 0x142F, 0x1428, 0x2853, 0x70B5, 0x70B3,
    0x70B1,
};

static const uint16_t bitalloc_65_codes_e[65] = {
    0x032C, 0x0332, 0x0378, 0x037E, 0x008C, 0x014A, 0x0188, 0x0197,
    0x019E, 0x01BD, 0x0044, 0x0047, 0x00AA, 0x00C5, 0x00CD, 0x00DC,
    0x001C, 0x002C, 0x0053, 0x0063, 0x0068, 0x0008, 0x000F, 0x0017,
    0x002B, 0x0035, 0x0005, 0x0009, 0x0016, 0x001C, 0x0006, 0x000F,
    0x0004, 0x0000, 0x0007, 0x001D, 0x0017, 0x000A, 0x0006, 0x0036,
    0x0030, 0x0028, 0x0010, 0x0009, 0x0069, 0x0064, 0x0054, 0x002D,
    0x001D, 0x00DD, 0x00CE, 0x00CA, 0x00AB, 0x00A4, 0x0045, 0x01BE,
    0x019F, 0x0198, 0x0189, 0x014B, 0x008D, 0x037F, 0x0379, 0x0333,
    0x032D,
};

static const uint16_t bitalloc_65_codes_f[65] = {
    0x0FE0, 0x0FE2, 0x0FE8, 0x0FEA, 0x0FEC, 0x0FEE, 0x0FF0, 0x0FF2,
    0x0FF4, 0x2FF2, 0x07F2, 0x07FB, 0x03F6, 0x0BFA, 0x0BFD, 0x01FF,
    0x05FF, 0x02FC, 0x007C, 0x017C, 0x003C, 0x00BC, 0x001C, 0x005C,
    0x000C, 0x002C, 0x0004, 0x0014, 0x0000, 0x0008, 0x000E, 0x0002,
    0x0006, 0x0003, 0x000F, 0x0009, 0x0001, 0x0015, 0x0005, 0x002D,
    0x000D, 0x005D, 0x001D, 0x00BD, 0x003D, 0x017D, 0x007D, 0x02FD,
    0x00FC, 0x05FC, 0x01FA, 0x0BFB, 0x03F7, 0x17F8, 0x07F3, 0x2FF3,
    0x0FF5, 0x0FF3, 0x0FF1, 0x0FEF, 0x0FED, 0x0FEB, 0x0FE9, 0x0FE3,
    0x0FE1,
};

static const uint16_t bitalloc_65_codes_g[65] = {
    0x010C, 0x038A, 0x0608, 0x0786, 0x0084, 0x0087, 0x0302, 0x0305,
    0x0040, 0x00E0, 0x00E3, 0x0183, 0x001E, 0x005E, 0x009E, 0x00DE,
    0x00F1, 0x0011, 0x0039, 0x0061, 0x0079, 0x0009, 0x001D, 0x0031,
    0x003D, 0x0005, 0x000F, 0x0019, 0x001F, 0x0003, 0x0006, 0x000A,
    0x000E, 0x000B, 0x0008, 0x0004, 0x0000, 0x001A, 0x0012, 0x000A,
    0x0002, 0x0036, 0x0026, 0x0016, 0x0006, 0x006E, 0x004E, 0x002E,
    0x000E, 0x00DF, 0x009F, 0x005F, 0x001F, 0x01E0, 0x0180, 0x00E1,
    0x0041, 0x03C2, 0x0303, 0x01C4, 0x0085, 0x0787, 0x0609, 0x038B,
    0x010D,
};

static const uint8_t bitalloc_65_bits_a[65] = {
    16, 16, 15, 14, 13, 13, 12, 12, 11, 11, 10, 10,  9,  9,  8,  8,
     7,  7,  7,  6,  6,  6,  6,  5,  5,  5,  5,  5,  5,  5,  4,  4,
     4,  4,  4,  4,  5,  5,  5,  5,  5,  5,  5,  6,  6,  6,  7,  7,
     7,  8,  8,  9,  9, 10, 10, 11, 11, 11, 12, 13, 13, 14, 15, 16,
    16,
};

static const uint8_t bitalloc_65_bits_b[65] = {
    12, 11, 10,  9,  9,  9,  8,  8,  8,  8,  8,  7,  7,  7,  7,  7,
     7,  6,  6,  6,  6,  6,  6,  6,  5,  5,  5,  5,  5,  5,  5,  4,
     4,  4,  5,  5,  5,  5,  5,  5,  5,  6,  6,  6,  6,  6,  6,  6,
     6,  7,  7,  7,  7,  7,  8,  8,  8,  8,  8,  9,  9,  9, 10, 10,
    12,
};

static const uint8_t bitalloc_65_bits_c[65] = {
    11, 10, 10,  9,  9,  8,  8,  8,  7,  7,  7,  7,  7,  7,  7,  6,
     6,  6,  6,  6,  6,  6,  6,  5,  5,  5,  5,  5,  5,  5,  5,  5,
     5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  6,  6,  6,  6,  6,  6,
     6,  6,  6,  7,  7,  7,  7,  7,  7,  8,  8,  8,  8,  9,  9, 10,
    11,
};

static const uint8_t bitalloc_65_bits_d[65] = {
    15, 15, 15, 14, 14, 13, 13, 12, 12, 11, 11, 10, 10, 10,  9,  9,
     9,  8,  8,  8,  7,  7,  7,  6,  6,  6,  5,  5,  5,  4,  4,  3,
     3,  3,  4,  4,  5,  5,  5,  6,  6,  6,  7,  7,  7,  8,  8,  8,
     8,  9,  9, 10, 10, 10, 11, 11, 12, 12, 13, 13, 13, 14, 15, 15,
    15,
};

static const uint8_t bitalloc_65_bits_e[65] = {
    10, 10, 10, 10,  9,  9,  9,  9,  9,  9,  8,  8,  8,  8,  8,  8,
     7,  7,  7,  7,  7,  6,  6,  6,  6,  6,  5,  5,  5,  5,  4,  4,
     3,  3,  4,  5,  5,  5,  5,  6,  6,  6,  6,  6,  7,  7,  7,  7,
     7,  8,  8,  8,  8,  8,  8,  9,  9,  9,  9,  9,  9, 10, 10, 10,
    10,
};

static const uint8_t bitalloc_65_bits_f[65] = {
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 13, 13, 12, 12, 12, 11,
    11, 10,  9,  9,  8,  8,  7,  7,  6,  6,  5,  5,  4,  4,  4,  3,
     3,  3,  4,  4,  4,  5,  5,  6,  6,  7,  7,  8,  8,  9,  9, 10,
    10, 11, 11, 12, 12, 13, 13, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14,
};

static const uint8_t bitalloc_65_bits_g[65] = {
    11, 11, 11, 11, 10, 10, 10, 10,  9,  9,  9,  9,  8,  8,  8,  8,
     8,  7,  7,  7,  7,  6,  6,  6,  6,  5,  5,  5,  5,  4,  4,  4,
     4,  4,  4,  4,  4,  5,  5,  5,  5,  6,  6,  6,  6,  7,  7,  7,
     7,  8,  8,  8,  8,  9,  9,  9,  9, 10, 10, 10, 10, 11, 11, 11,
    11,
};

static const uint16_t bitalloc_129_codes_a[129] = {
    0x0660, 0x0666, 0x06EC, 0x0722, 0x0760, 0x076E, 0x004C, 0x004E,
    0x00F4, 0x010A, 0x0148, 0x0156, 0x01D4, 0x01F2, 0x0331, 0x0370,
    0x0377, 0x0396, 0x03B1, 0x0024, 0x0064, 0x007B, 0x008A, 0x00A5,
    0x00D4, 0x00EB, 0x00FA, 0x019A, 0x01B9, 0x01C9, 0x01D9, 0x0010,
    0x0030, 0x0033, 0x0043, 0x0053, 0x006B, 0x007A, 0x00CA, 0x00D2,
    0x00DE, 0x00E6, 0x00F6, 0x000E, 0x001F, 0x0023, 0x002B, 0x003B,
    0x003F, 0x0067, 0x0070, 0x0077, 0x0005, 0x000D, 0x0012, 0x001B,
    0x002C, 0x0035, 0x003A, 0x0004, 0x000B, 0x0017, 0x001F, 0x0009,
    0x0008, 0x000A, 0x0000, 0x0018, 0x000C, 0x0005, 0x003C, 0x0036,
    0x002D, 0x001C, 0x0013, 0x000E, 0x0006, 0x007A, 0x0071, 0x0068,
    0x0064, 0x003C, 0x0034, 0x0028, 0x0020, 0x000F, 0x00F7, 0x00E7,
    0x00DF, 0x00D3, 0x00CB, 0x007B, 0x0074, 0x0054, 0x0044, 0x003C,
    0x0031, 0x0011, 0x01DA, 0x01CA, 0x01BA, 0x019B, 0x00FB, 0x00F8,
    0x00D5, 0x00AA, 0x008B, 0x0084, 0x0065, 0x0025, 0x03B6, 0x0397,
    0x0390, 0x0371, 0x0332, 0x01F3, 0x01D5, 0x0157, 0x0149, 0x010B,
    0x00F5, 0x004F, 0x004D, 0x076F, 0x0761, 0x0723, 0x06ED, 0x0667,
    0x0661,
};

static const uint16_t bitalloc_129_codes_b[129] = {
    0x29DC, 0x14EF, 0x0455, 0x0E9C, 0x022B, 0x0489, 0x0740, 0x074F,
    0x0172, 0x0245, 0x0247, 0x030A, 0x03A1, 0x001C, 0x008B, 0x00D6,
    0x010C, 0x0148, 0x014F, 0x0186, 0x01D1, 0x0008, 0x000F, 0x0046,
    0x005D, 0x0078, 0x0087, 0x0096, 0x00A5, 0x00BC, 0x00D8, 0x00DE,
    0x00F6, 0x0005, 0x0014, 0x0024, 0x002F, 0x003A, 0x003D, 0x0049,
    0x0050, 0x0058, 0x005F, 0x0066, 0x006D, 0x0075, 0x007C, 0x0004,
    0x000B, 0x0013, 0x0018, 0x001B, 0x001F, 0x0022, 0x0026, 0x002A,
    0x002D, 0x0031, 0x0034, 0x0038, 0x003B, 0x003F, 0x0003, 0x0006,
    0x000A, 0x0007, 0x0004, 0x0000, 0x003C, 0x0039, 0x0035, 0x0032,
    0x002E, 0x002B, 0x0027, 0x0023, 0x0020, 0x001C, 0x0019, 0x0016,
    0x0010, 0x0005, 0x007D, 0x007A, 0x006E, 0x0067, 0x0060, 0x0059,
    0x0051, 0x004A, 0x0042, 0x003B, 0x0034, 0x0025, 0x0015, 0x0006,
    0x00F7, 0x00DF, 0x00D9, 0x00BD, 0x00A6, 0x0097, 0x0090, 0x0079,
    0x006A, 0x0047, 0x0044, 0x0009, 0x01D2, 0x0187, 0x0184, 0x0149,
    0x010D, 0x00D7, 0x00B8, 0x001D, 0x03A6, 0x030B, 0x029C, 0x0246,
    0x0173, 0x0114, 0x0741, 0x053A, 0x0488, 0x0E9D, 0x0A76, 0x0454,
    0x29DD,
};

static const uint16_t bitalloc_129_codes_c[129] = {
    0x0E5C, 0x072F, 0x001D, 0x0724, 0x000F, 0x010D, 0x0324, 0x0393,
    0x03E9, 0x0080, 0x0087, 0x00FA, 0x0164, 0x0193, 0x01DE, 0x01F5,
    0x0010, 0x002A, 0x0041, 0x0064, 0x0073, 0x008E, 0x00A4, 0x00B3,
    0x00D6, 0x00E5, 0x00F4, 0x00FB, 0x0002, 0x0009, 0x0013, 0x001E,
    0x0026, 0x002C, 0x0033, 0x003F, 0x0041, 0x004C, 0x0053, 0x005E,
    0x0065, 0x0070, 0x0073, 0x0078, 0x007B, 0x007E, 0x0002, 0x0005,
    0x0007, 0x000B, 0x000D, 0x0011, 0x0014, 0x0017, 0x001A, 0x001D,
    0x0021, 0x0024, 0x0027, 0x002A, 0x002D, 0x0030, 0x0033, 0x0036,
    0x003A, 0x0037, 0x0034, 0x0031, 0x002E, 0x002B, 0x0028, 0x0025,
    0x0022, 0x001E, 0x001B, 0x0018, 0x0015, 0x0012, 0x000E, 0x000C,
    0x0008, 0x0006, 0x0003, 0x007F, 0x007C, 0x0079, 0x0076, 0x0071,
    0x006A, 0x005F, 0x0058, 0x004D, 0x0046, 0x0040, 0x0038, 0x002D,
    0x0027, 0x001F, 0x0014, 0x0012, 0x0003, 0x0000, 0x00F5, 0x00EE,
    0x00D7, 0x00C8, 0x00A5, 0x008F, 0x007C, 0x0065, 0x0042, 0x002B,
    0x0011, 0x0002, 0x01DF, 0x01C8, 0x0165, 0x00FB, 0x00E4, 0x0081,
    0x0006, 0x03E8, 0x0325, 0x01CA, 0x010C, 0x0725, 0x0396, 0x001C,
    0x0E5D,
};

static const uint16_t bitalloc_129_codes_d[129] = {
    0xA598, 0xA59A, 0xA59C, 0xA59E, 0xC598, 0xE586, 0x3ACC, 0x52CA,
    0x62CD, 0x0D48, 0x1D67, 0x2978, 0x3167, 0x3966, 0x06A5, 0x0EBC,
    0x14BD, 0x1CB1, 0x0350, 0x0353, 0x075F, 0x0A5F, 0x0C5E, 0x0E5E,
    0x01AE, 0x03AD, 0x052D, 0x062D, 0x072D, 0x00D5, 0x01D4, 0x0294,
    0x0314, 0x0394, 0x0014, 0x0094, 0x0114, 0x0174, 0x01B4, 0x01F4,
    0x000B, 0x004B, 0x008B, 0x00BB, 0x00DB, 0x00FB, 0x001B, 0x003B,
    0x0053, 0x0063, 0x0073, 0x0003, 0x0013, 0x0023, 0x002F, 0x0037,
    0x003F, 0x0007, 0x000F, 0x0015, 0x0019, 0x001D, 0x0001, 0x0005,
    0x0009, 0x0006, 0x0002, 0x001E, 0x001A, 0x0016, 0x0010, 0x0008,
    0x0000, 0x0038, 0x0030, 0x0028, 0x001C, 0x000C, 0x007C, 0x006C,
    0x005C, 0x0044, 0x0024, 0x0004, 0x00E4, 0x00C4, 0x00A4, 0x0074,
    0x0034, 0x01F5, 0x01B5, 0x0175, 0x0115, 0x0095, 0x0015, 0x0395,
    0x0315, 0x0295, 0x01D5, 0x00D6, 0x072E, 0x062E, 0x052E, 0x03AE,
    0x01AF, 0x0E5F, 0x0C5F, 0x0C58, 0x0A58, 0x0758, 0x0351, 0x1CB2,
    0x18B2, 0x0EBD, 0x0EB2, 0x3967, 0x3960, 0x2979, 0x2964, 0x0D49,
    0x72C2, 0x52CB, 0x3ACD, 0xE587, 0xC599, 0xA59F, 0xA59D, 0xA59B,
    0xA599,
};

static const uint16_t bitalloc_129_codes_e[129] = {
    0xA13C, 0xC720, 0xA13F, 0xA13E, 0xA13D, 0xE722, 0x5090, 0x6393,
    0x7392, 0x2849, 0x31CE, 0x39CE, 0x1425, 0x18E5, 0x1CE5, 0x0844,
    0x0A1C, 0x0C7C, 0x036C, 0x0423, 0x050F, 0x063F, 0x01B7, 0x0216,
    0x0285, 0x031D, 0x039D, 0x0109, 0x0140, 0x0180, 0x01C8, 0x01CF,
    0x007A, 0x008A, 0x00A2, 0x00C1, 0x00E5, 0x0014, 0x0037, 0x0043,
    0x004E, 0x0056, 0x0061, 0x006C, 0x007C, 0x000B, 0x001C, 0x001F,
    0x0023, 0x0025, 0x0029, 0x002C, 0x002E, 0x0032, 0x0034, 0x0037,
    0x003A, 0x003C, 0x003F, 0x0001, 0x0003, 0x0006, 0x0008, 0x000A,
    0x000C, 0x000B, 0x0009, 0x0007, 0x0004, 0x0002, 0x0000, 0x003D,
    0x003B, 0x0038, 0x0035, 0x0033, 0x002F, 0x002D, 0x002A, 0x0026,
    0x0024, 0x0020, 0x001D, 0x001A, 0x007D, 0x006D, 0x0062, 0x0057,
    0x004F, 0x0044, 0x003C, 0x0015, 0x00E6, 0x00C6, 0x00A3, 0x008B,
    0x007B, 0x006C, 0x01C9, 0x0181, 0x0141, 0x010A, 0x00DA, 0x031E,
    0x0286, 0x0217, 0x0210, 0x0738, 0x0638, 0x0508, 0x036D, 0x0C7D,
    0x0A1D, 0x0845, 0x1CE6, 0x18E6, 0x1426, 0x39CF, 0x31CF, 0x284E,
    0x7393, 0x7390, 0x5091, 0xE723, 0xC724, 0xC725, 0xC722, 0xC723,
    0xC721,
};

static const uint16_t bitalloc_129_codes_f[129] = {
    0x762C, 0x3B17, 0x1555, 0x0608, 0x0AAB, 0x0FF2, 0x0305, 0x0307,
    0x0763, 0x0046, 0x010C, 0x01BC, 0x02AB, 0x03B6, 0x03FD, 0x0080,
    0x0087, 0x00DF, 0x0156, 0x01D9, 0x01F8, 0x01FF, 0x002A, 0x0041,
    0x0061, 0x0094, 0x00D4, 0x00EA, 0x00F2, 0x00FD, 0x0009, 0x000B,
    0x001A, 0x0026, 0x0031, 0x0040, 0x004B, 0x006B, 0x0073, 0x0077,
    0x007A, 0x007C, 0x0000, 0x0002, 0x0006, 0x0008, 0x000B, 0x000E,
    0x0011, 0x0014, 0x0016, 0x0019, 0x001C, 0x001E, 0x0021, 0x0023,
    0x0026, 0x0028, 0x002B, 0x002D, 0x002F, 0x0031, 0x0033, 0x0036,
    0x0038, 0x0037, 0x0034, 0x0032, 0x0030, 0x002E, 0x002C, 0x0029,
    0x0027, 0x0024, 0x0022, 0x001F, 0x001D, 0x001A, 0x0017, 0x0015,
    0x0012, 0x000F, 0x000C, 0x0009, 0x0007, 0x0003, 0x0001, 0x007D,
    0x007B, 0x0078, 0x0074, 0x0072, 0x0054, 0x0041, 0x0036, 0x0027,
    0x001B, 0x0014, 0x000A, 0x00FE, 0x00F3, 0x00EB, 0x00D5, 0x0095,
    0x006E, 0x0042, 0x002B, 0x0010, 0x01F9, 0x01DA, 0x0157, 0x0154,
    0x00C0, 0x0081, 0x0022, 0x03B7, 0x03B0, 0x01BD, 0x010D, 0x0047,
    0x07F8, 0x0554, 0x0306, 0x0FF3, 0x0EC4, 0x0609, 0x1D8A, 0x1554,
    0x762D,
};

static const uint16_t bitalloc_129_codes_g[129] = {
    0x1E20, 0x1E5E, 0x031C, 0x051A, 0x0718, 0x0916, 0x0B14, 0x0D12,
    0x0F11, 0x0090, 0x018F, 0x028E, 0x038D, 0x048C, 0x058B, 0x068A,
    0x0789, 0x0049, 0x00C8, 0x0148, 0x01C7, 0x0247, 0x02C6, 0x0346,
    0x03C5, 0x0025, 0x0065, 0x00A5, 0x00E4, 0x0124, 0x0164, 0x01A4,
    0x01E3, 0x0013, 0x0033, 0x0053, 0x0073, 0x0093, 0x00B3, 0x00D3,
    0x00F3, 0x000A, 0x001A, 0x002A, 0x003A, 0x004A, 0x005A, 0x006A,
    0x007A, 0x0006, 0x000E, 0x0016, 0x001E, 0x0026, 0x002E, 0x0036,
    0x003E, 0x0004, 0x0008, 0x000C, 0x0010, 0x0014, 0x0018, 0x001C,
    0x0000, 0x001D, 0x0019, 0x0015, 0x0011, 0x000D, 0x0009, 0x0005,
    0x003F, 0x0037, 0x002F, 0x0027, 0x001F, 0x0017, 0x000F, 0x0007,
    0x007B, 0x006B, 0x005B, 0x004B, 0x003B, 0x002B, 0x001B, 0x000B,
    0x0008, 0x00F0, 0x00D0, 0x00B0, 0x0090, 0x0070, 0x0050, 0x0030,
    0x01E4, 0x01A5, 0x0165, 0x0125, 0x00E5, 0x00E2, 0x00A2, 0x0062,
    0x03CA, 0x0347, 0x02C7, 0x02C4, 0x0244, 0x0149, 0x00C9, 0x00C6,
    0x0796, 0x068B, 0x0688, 0x048D, 0x048A, 0x028F, 0x028C, 0x0091,
    0x0F2E, 0x0D13, 0x0B15, 0x0917, 0x0719, 0x051B, 0x031D, 0x1E5F,
    0x1E21,
};

static const uint8_t bitalloc_129_bits_a[129] = {
    11, 11, 11, 11, 11, 11, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  8,
     8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  7,  7,  7,  7,  7,
     7,  7,  7,  7,  6,  6,  6,  6,  6,  6,  6,  5,  5,  5,  5,  4,
     4,  4,  4,  5,  5,  5,  6,  6,  6,  6,  6,  6,  6,  7,  7,  7,
     7,  7,  7,  7,  7,  7,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
     8,  8,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 11, 11, 11, 11, 11,
    11,
};

static const uint8_t bitalloc_129_bits_b[129] = {
    14, 13, 12, 12, 11, 11, 11, 11, 10, 10, 10, 10, 10,  9,  9,  9,
     9,  9,  9,  9,  9,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
     8,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  6,
     6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  5,  5,
     5,  5,  5,  5,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,
     6,  6,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,
     8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  9,  9,  9,  9,
     9,  9,  9,  9, 10, 10, 10, 10, 10, 10, 11, 11, 11, 12, 12, 12,
    14,
};

static const uint8_t bitalloc_129_bits_c[129] = {
    13, 12, 11, 11, 10, 10, 10, 10, 10,  9,  9,  9,  9,  9,  9,  9,
     8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  7,  7,  7,  7,
     7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  6,  6,
     6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,
     6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,
     6,  6,  6,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,
     7,  7,  7,  7,  7,  7,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
     8,  8,  9,  9,  9,  9,  9,  9,  9, 10, 10, 10, 10, 11, 11, 11,
    13,
};

static const uint8_t bitalloc_129_bits_d[129] = {
    16, 16, 16, 16, 16, 16, 15, 15, 15, 14, 14, 14, 14, 14, 13, 13,
    13, 13, 12, 12, 12, 12, 12, 12, 11, 11, 11, 11, 11, 10, 10, 10,
    10, 10,  9,  9,  9,  9,  9,  9,  8,  8,  8,  8,  8,  8,  7,  7,
     7,  7,  7,  6,  6,  6,  6,  6,  6,  5,  5,  5,  5,  5,  4,  4,
     4,  4,  4,  5,  5,  5,  5,  5,  5,  6,  6,  6,  6,  6,  7,  7,
     7,  7,  7,  7,  8,  8,  8,  8,  8,  9,  9,  9,  9,  9,  9, 10,
    10, 10, 10, 10, 11, 11, 11, 11, 11, 12, 12, 12, 12, 12, 12, 13,
    13, 13, 13, 14, 14, 14, 14, 14, 15, 15, 15, 16, 16, 16, 16, 16,
    16,
};

static const uint8_t bitalloc_129_bits_e[129] = {
    16, 16, 16, 16, 16, 16, 15, 15, 15, 14, 14, 14, 13, 13, 13, 12,
    12, 12, 11, 11, 11, 11, 10, 10, 10, 10, 10,  9,  9,  9,  9,  9,
     8,  8,  8,  8,  8,  7,  7,  7,  7,  7,  7,  7,  7,  6,  6,  6,
     6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  5,  5,  5,  5,  5,
     5,  5,  5,  5,  5,  5,  5,  6,  6,  6,  6,  6,  6,  6,  6,  6,
     6,  6,  6,  6,  7,  7,  7,  7,  7,  7,  7,  7,  8,  8,  8,  8,
     8,  8,  9,  9,  9,  9,  9, 10, 10, 10, 10, 11, 11, 11, 11, 12,
    12, 12, 13, 13, 13, 14, 14, 14, 15, 15, 15, 16, 16, 16, 16, 16,
    16,
};

static const uint8_t bitalloc_129_bits_f[129] = {
    15, 14, 13, 12, 12, 12, 11, 11, 11, 10, 10, 10, 10, 10, 10,  9,
     9,  9,  9,  9,  9,  9,  8,  8,  8,  8,  8,  8,  8,  8,  7,  7,
     7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  6,  6,  6,  6,  6,  6,
     6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,
     6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,
     6,  6,  6,  6,  6,  6,  6,  7,  7,  7,  7,  7,  7,  7,  7,  7,
     7,  7,  7,  8,  8,  8,  8,  8,  8,  8,  8,  8,  9,  9,  9,  9,
     9,  9,  9, 10, 10, 10, 10, 10, 11, 11, 11, 12, 12, 12, 13, 13,
    15,
};

static const uint8_t bitalloc_129_bits_g[129] = {
    13, 13, 12, 12, 12, 12, 12, 12, 12, 11, 11, 11, 11, 11, 11, 11,
    11, 10, 10, 10, 10, 10, 10, 10, 10,  9,  9,  9,  9,  9,  9,  9,
     9,  8,  8,  8,  8,  8,  8,  8,  8,  7,  7,  7,  7,  7,  7,  7,
     7,  6,  6,  6,  6,  6,  6,  6,  6,  5,  5,  5,  5,  5,  5,  5,
     4,  5,  5,  5,  5,  5,  5,  5,  6,  6,  6,  6,  6,  6,  6,  6,
     7,  7,  7,  7,  7,  7,  7,  7,  7,  8,  8,  8,  8,  8,  8,  8,
     9,  9,  9,  9,  9,  9,  9,  9, 10, 10, 10, 10, 10, 10, 10, 10,
    11, 11, 11, 11, 11, 11, 11, 11, 12, 12, 12, 12, 12, 12, 12, 13,
    13,
};

static const uint8_t bitalloc_sizes[DCA_CODE_BOOKS] = {
    3, 5, 7, 9, 13, 17, 25, 33, 65, 129
};

static const int8_t bitalloc_offsets[DCA_CODE_BOOKS] = {
    -1, -2, -3, -4, -6, -8, -12, -16, -32, -64
};

static const uint8_t bitalloc_maxbits[DCA_CODE_BOOKS][7] = {
    { 2 },
    { 4, 3, 3 },
    { 5, 5, 4 },
    { 6, 5, 6 },
    { 7, 6, 5 },
    { 9, 8, 7, 9, 8, 8, 8 },
    { 9, 9, 8, 9, 8, 9, 9 },
    { 9, 9, 9, 9, 9, 9, 9 },
    { 9, 9, 9, 9, 9, 9, 9 },
    { 9, 9, 9, 9, 9, 9, 9 }
};

static const uint16_t *const bitalloc_codes[DCA_CODE_BOOKS][8] = {
    { bitalloc_3_codes,     NULL },
    { bitalloc_5_codes_a,   bitalloc_5_codes_b,   bitalloc_5_codes_c,   NULL },
    { bitalloc_7_codes_a,   bitalloc_7_codes_b,   bitalloc_7_codes_c,   NULL },
    { bitalloc_9_codes_a,   bitalloc_9_codes_b,   bitalloc_9_codes_c,   NULL },
    { bitalloc_13_codes_a,  bitalloc_13_codes_b,  bitalloc_13_codes_c,  NULL },
    { bitalloc_17_codes_a,  bitalloc_17_codes_b,  bitalloc_17_codes_c,  bitalloc_17_codes_d,
      bitalloc_17_codes_e,  bitalloc_17_codes_f,  bitalloc_17_codes_g,  NULL },
    { bitalloc_25_codes_a,  bitalloc_25_codes_b,  bitalloc_25_codes_c,  bitalloc_25_codes_d,
      bitalloc_25_codes_e,  bitalloc_25_codes_f,  bitalloc_25_codes_g,  NULL },
    { bitalloc_33_codes_a,  bitalloc_33_codes_b,  bitalloc_33_codes_c,  bitalloc_33_codes_d,
      bitalloc_33_codes_e,  bitalloc_33_codes_f,  bitalloc_33_codes_g,  NULL },
    { bitalloc_65_codes_a,  bitalloc_65_codes_b,  bitalloc_65_codes_c,  bitalloc_65_codes_d,
      bitalloc_65_codes_e,  bitalloc_65_codes_f,  bitalloc_65_codes_g,  NULL },
    { bitalloc_129_codes_a, bitalloc_129_codes_b, bitalloc_129_codes_c, bitalloc_129_codes_d,
      bitalloc_129_codes_e, bitalloc_129_codes_f, bitalloc_129_codes_g, NULL }
};

static const uint8_t *const bitalloc_bits[DCA_CODE_BOOKS][8] = {
    { bitalloc_3_bits,     NULL },
    { bitalloc_5_bits_a,   bitalloc_5_bits_b,   bitalloc_5_bits_c,   NULL },
    { bitalloc_7_bits_a,   bitalloc_7_bits_b,   bitalloc_7_bits_c,   NULL },
    { bitalloc_9_bits_a,   bitalloc_9_bits_b,   bitalloc_9_bits_c,   NULL },
    { bitalloc_13_bits_a,  bitalloc_13_bits_b,  bitalloc_13_bits_c,  NULL },
    { bitalloc_17_bits_a,  bitalloc_17_bits_b,  bitalloc_17_bits_c,  bitalloc_17_bits_d,
      bitalloc_17_bits_e,  bitalloc_17_bits_f,  bitalloc_17_bits_g,  NULL },
    { bitalloc_25_bits_a,  bitalloc_25_bits_b,  bitalloc_25_bits_c,  bitalloc_25_bits_d,
      bitalloc_25_bits_e,  bitalloc_25_bits_f,  bitalloc_25_bits_g,  NULL },
    { bitalloc_33_bits_a,  bitalloc_33_bits_b,  bitalloc_33_bits_c,  bitalloc_33_bits_d,
      bitalloc_33_bits_e,  bitalloc_33_bits_f,  bitalloc_33_bits_g,  NULL },
    { bitalloc_65_bits_a,  bitalloc_65_bits_b,  bitalloc_65_bits_c,  bitalloc_65_bits_d,
      bitalloc_65_bits_e,  bitalloc_65_bits_f,  bitalloc_65_bits_g,  NULL },
    { bitalloc_129_bits_a, bitalloc_129_bits_b, bitalloc_129_bits_c, bitalloc_129_bits_d,
      bitalloc_129_bits_e, bitalloc_129_bits_f, bitalloc_129_bits_g, NULL }
};

static const uint16_t tnl_grp_0_codes[37] = {
    0x0000, 0x0003, 0x0004, 0x0007, 0x0001, 0x0009, 0x000a, 0x000d,
    0x000e, 0x0006, 0x0012, 0x0005, 0x0015, 0x0016, 0x0022, 0x0025,
    0x0035, 0x0076, 0x0002, 0x0042, 0x00b6, 0x0036, 0x00c2, 0x0136,
    0x0182, 0x01c2, 0x03c2, 0x0482, 0x0682, 0x0082, 0x0882, 0x0a82,
    0x0282, 0x2282, 0x3282, 0x1282, 0x5282,
};

static const uint16_t tnl_grp_1_codes[34] = {
    0x0001, 0x0003, 0x0006, 0x0000, 0x0002, 0x0004, 0x0005, 0x0007,
    0x0008, 0x000f, 0x001a, 0x001c, 0x001d, 0x000a, 0x002c, 0x002d,
    0x000d, 0x002a, 0x004c, 0x004d, 0x006a, 0x008c, 0x00cd, 0x00ea,
    0x000c, 0x010c, 0x01ea, 0x020c, 0x030c, 0x07ea, 0x0bea, 0x03ea,
    0x13ea, 0x33ea,
};

static const uint16_t tnl_grp_2_codes[31] = {
    0x0001, 0x0003, 0x0006, 0x0007, 0x0004, 0x0008, 0x000c, 0x0010,
    0x0012, 0x001a, 0x0022, 0x0000, 0x000a, 0x0020, 0x0040, 0x004a,
    0x006a, 0x0002, 0x002a, 0x0042, 0x0082, 0x00aa, 0x00e0, 0x0060,
    0x00c2, 0x01c2, 0x0160, 0x0360, 0x0f60, 0x0760, 0x1760,
};

static const uint16_t tnl_grp_3_codes[28] = {
    0x0001, 0x0006, 0x0008, 0x0014, 0x001c, 0x0000, 0x0002, 0x0004,
    0x000a, 0x000c, 0x0010, 0x0012, 0x001a, 0x0020, 0x002a, 0x002c,
    0x0032, 0x003a, 0x0022, 0x0030, 0x0062, 0x0064, 0x0070, 0x0024,
    0x00a4, 0x01a4, 0x03a4, 0x07a4,
};

static const uint16_t tnl_grp_4_codes[23] = {
    0x0001, 0x0000, 0x000a, 0x0006, 0x0012, 0x001e, 0x0022, 0x002e,
    0x0036, 0x003e, 0x0002, 0x0016, 0x0032, 0x004e, 0x0056, 0x000e,
    0x0042, 0x0072, 0x00c2, 0x00f2, 0x008e, 0x018e, 0x038e,
};

static const uint16_t tnl_scf_codes[20] = {
    0x0000, 0x0001, 0x0002, 0x0005, 0x0006, 0x0007, 0x000b, 0x000c,
    0x0013, 0x0014, 0x0003, 0x0004, 0x0023, 0x0064, 0x00a4, 0x0024,
    0x0124, 0x0324, 0x0724, 0x0f24,
};

static const uint16_t damp_codes[7] = {
    0x0001, 0x0000, 0x0002, 0x0006, 0x000e, 0x001e, 0x003e,
};

static const uint16_t dph_codes[9] = {
    0x0000, 0x0002, 0x0003, 0x0001, 0x0009, 0x000d, 0x0005, 0x0015,
    0x0035,
};

static const uint16_t fst_rsd_amp_codes[24] = {
    0x0003, 0x0005, 0x0006, 0x0007, 0x0000, 0x0001, 0x0002, 0x0008,
    0x0009, 0x000a, 0x0014, 0x0004, 0x001a, 0x001c, 0x0024, 0x002c,
    0x003a, 0x000c, 0x003c, 0x004c, 0x00fc, 0x007c, 0x017c, 0x037c,
};

static const uint16_t rsd_apprx_codes[6] = {
    0x0000, 0x0001, 0x0003, 0x0007, 0x000f, 0x001f,
};

static const uint16_t rsd_amp_codes[33] = {
    0x0001, 0x0000, 0x0002, 0x0003, 0x0004, 0x000e, 0x000f, 0x0016,
    0x0007, 0x0027, 0x0037, 0x0026, 0x0066, 0x0006, 0x0017, 0x0046,
    0x0097, 0x00d7, 0x0086, 0x00c6, 0x01c6, 0x0157, 0x0186, 0x0257,
    0x0357, 0x0057, 0x0786, 0x0386, 0x0b86, 0x0457, 0x0c57, 0x1457,
    0x1c57,
};

static const uint16_t avg_g3_codes[18] = {
    0x0001, 0x0002, 0x0003, 0x0000, 0x000c, 0x0014, 0x0018, 0x0004,
    0x0008, 0x0028, 0x0068, 0x0024, 0x00a4, 0x00e4, 0x0164, 0x0064,
    0x0264, 0x0664,
};

static const uint16_t st_grid_codes[22] = {
    0x0001, 0x0002, 0x0000, 0x0004, 0x0008, 0x001c, 0x004c, 0x006c,
    0x000c, 0x002c, 0x008c, 0x00ac, 0x012c, 0x018c, 0x01ac, 0x038c,
    0x03ac, 0x032c, 0x072c, 0x0f2c, 0x172c, 0x1f2c,
};

static const uint16_t grid_2_codes[20] = {
    0x0000, 0x0002, 0x0003, 0x0001, 0x0005, 0x000d, 0x003d, 0x005d,
    0x009d, 0x011d, 0x001d, 0x061d, 0x041d, 0x0c1d, 0x0a1d, 0x121d,
    0x021d, 0x1a1d, 0x221d, 0x3a1d,
};

static const uint16_t grid_3_codes[13] = {
    0x0001, 0x0002, 0x0000, 0x0004, 0x000c, 0x001c, 0x007c, 0x003c,
    0x01bc, 0x00bc, 0x06bc, 0x02bc, 0x0abc,
};

static const uint16_t rsd_codes[9] = {
    0x0001, 0x0003, 0x0000, 0x0002, 0x0006, 0x0004, 0x000c, 0x001c,
    0x003c,
};

static const uint8_t tnl_grp_0_bitvals[74] = {
     3,  5,  3,  9,  3,  4,  3,  6,  4, 10,  4, 13,  4,  7,  4, 11,
     4,  8,  5, 12,  5, 14,  6, 15,  6, 18,  6,  1,  6, 17,  6, 16,
     6, 21,  7, 20,  8, 19,  8, 22,  8, 25,  9, 26,  9, 23,  9,  3,
     9, 24, 10, 29, 10, 27, 11, 28, 11, 30, 12, 33, 12, 31, 12, 32,
    14, 34, 14, 37, 14, 36, 15, 35, 15,  0,
};

static const uint8_t tnl_grp_1_bitvals[68] = {
     3,  9,  3,  6,  3,  5,  4,  4,  4,  8,  4, 10,  4,  1,  4, 11,
     4,  7,  4, 13,  5, 12,  5, 14,  5, 17,  6, 16,  6, 15,  6, 18,
     7, 20,  7, 19,  7, 21,  8, 25,  8, 23,  8, 22,  8, 24,  9, 26,
    10,  3, 10, 29, 10, 30, 10, 27, 10, 28, 11, 31, 12, 32, 13, 33,
    14, 34, 14,  0,
};

static const uint8_t tnl_grp_2_bitvals[62] = {
     2,  1,  3,  6,  3,  5,  3,  7,  4,  9,  4,  8,  4,  4,  5, 10,
     5, 11,  5, 13,  6, 12,  7, 14,  7, 16,  7, 15,  7, 17,  7, 18,
     7, 19,  8, 22,  8, 20,  8, 21,  8,  3,  8, 24,  8, 25,  9, 23,
     9, 26,  9, 27, 10, 28, 11, 29, 12, 31, 13, 30, 13,  0,
};

static const uint8_t tnl_grp_3_bitvals[56] = {
     1,  1,  3,  6,  4,  5,  5,  9,  5,  4,  6,  8,  6, 14,  6, 10,
     6, 21,  6, 13,  6,  7,  6,  3,  6, 16,  6,  2,  6, 18,  6, 17,
     6, 11,  6, 15,  7, 19,  7, 23,  7, 24,  7, 22,  7, 12,  8, 20,
     9, 25, 10, 26, 11, 27, 11,  0,
};

static const uint8_t tnl_grp_4_bitvals[46] = {
     1,  1,  2,  2,  4,  4,  5,  5,  6,  6,  6,  8,  6,  3,  6, 19,
     6, 20,  6,  9,  7,  7,  7, 11,  7, 13,  7, 17,  7, 10,  8, 12,
     8, 15,  8, 14,  8, 21,  8, 18,  9, 16, 10, 22, 10,  0,
};

static const uint8_t tnl_scf_bitvals[40] = {
     3,  3,  3,  1,  3,  2,  3,  5,  3,  4,  3,  6,  4,  8,  4,  7,
     5, 10,  5,  9,  6, 12,  6, 11,  6, 13,  7, 14,  8, 15,  9, 16,
    10, 17, 11, 18, 12, 19, 12,  0,
};

static const uint8_t damp_bitvals[14] = {
     1,  1,  2,  2,  3,  3,  4,  4,  5,  5,  6,  6,  6,  0,
};

static const uint8_t dph_bitvals[18] = {
     2,  2,  2,  1,  2,  8,  4,  3,  4,  7,  4,  4,  5,  6,  6,  5,
     6,  0,
};

static const uint8_t fst_rsd_amp_bitvals[48] = {
     3, 13,  3, 15,  3, 16,  3, 14,  4, 12,  4, 10,  4, 11,  4, 17,
     4, 18,  5, 19,  5,  9,  6,  1,  6,  7,  6,  6,  6,  8,  6,  5,
     6,  4,  7, 20,  7,  2,  7,  3,  8, 21,  9, 22, 10, 23, 10,  0,
};

static const uint8_t rsd_apprx_bitvals[12] = {
     1,  1,  2,  2,  3,  3,  4,  4,  5,  5,  5,  0,
};

static const uint8_t rsd_amp_bitvals[66] = {
     2,  3,  3,  2,  3,  5,  3,  4,  3,  1,  4,  7,  4,  6,  5,  9,
     6,  8,  6, 11,  6, 10,  7, 12,  7, 13,  8, 14,  8, 18,  8, 16,
     8, 15,  8, 22,  9, 20,  9, 24,  9, 17, 10, 28, 10, 26, 10, 21,
    10, 23, 11, 30, 11, 19, 12, 25, 12, 32, 13, 36, 13, 29, 13, 34,
    13,  0,
};

static const uint8_t avg_g3_bitvals[36] = {
     2, 15,  2, 16,  2, 17,  4, 14,  4, 18,  5, 12,  5, 13,  6, 10,
     6, 11,  7, 19,  7,  9,  8, 20,  8,  8,  8,  7,  9, 21, 10,  6,
    11, 23, 11,  0,
};

static const uint8_t st_grid_bitvals[44] = {
     1,  6,  2,  1,  4,  4,  4,  8,  4,  3,  5, 10,  7, 12,  7,  5,
     8, 14,  9, 16,  9,  7,  9, 18, 10, 11, 10,  9, 10, 20, 10, 22,
    10,  2, 11, 13, 13, 17, 13, 24, 13, 15, 13,  0,
};

static const uint8_t grid_2_bitvals[40] = {
     2,  3,  2,  2,  2,  1,  3,  4,  4,  5,  5,  6,  6,  7,  7,  8,
     8,  9,  9, 10, 11, 11, 11, 12, 12, 13, 12, 17, 13, 15, 13, 18,
    14, 19, 14, 16, 14, 14, 14,  0,
};

static const uint8_t grid_3_bitvals[26] = {
     1, 17,  2, 16,  3, 18,  4, 15,  5, 19,  6, 14,  7, 20,  8, 13,
     9, 21, 10, 12, 11, 22, 12, 11, 12,  0,
};

static const uint8_t rsd_bitvals[18] = {
     2,  2,  2,  3,  3,  1,  3,  4,  3,  0,  4,  5,  5,  6,  6,  7,
     6,  4,
};

static const uint16_t vlc_offs[80] = {
        0,   512,   640,   768,  1282,  1794,  2436,  3080,  3770,  4454,  5364,
     5372,  5380,  5388,  5392,  5396,  5412,  5420,  5428,  5460,  5492,  5508,
     5572,  5604,  5668,  5796,  5860,  5892,  6412,  6668,  6796,  7308,  7564,
     7820,  8076,  8620,  9132,  9388,  9910, 10166, 10680, 11196, 11726, 12240,
    12752, 13298, 13810, 14326, 14840, 15500, 16022, 16540, 17158, 17678, 18264,
    18796, 19352, 19926, 20468, 21472, 22398, 23014, 23622, 24200, 24748, 25276,
    25792, 26306, 26826, 26890, 26954, 27468, 27500, 28038, 28554, 29086, 29630,
    30150, 30214
};

DCAVLC  ff_dca_vlc_bit_allocation;
DCAVLC  ff_dca_vlc_transition_mode;
DCAVLC  ff_dca_vlc_scale_factor;
DCAVLC  ff_dca_vlc_quant_index[DCA_CODE_BOOKS];

VLC     ff_dca_vlc_tnl_grp[5];
VLC     ff_dca_vlc_tnl_scf;
VLC     ff_dca_vlc_damp;
VLC     ff_dca_vlc_dph;
VLC     ff_dca_vlc_fst_rsd_amp;
VLC     ff_dca_vlc_rsd_apprx;
VLC     ff_dca_vlc_rsd_amp;
VLC     ff_dca_vlc_avg_g3;
VLC     ff_dca_vlc_st_grid;
VLC     ff_dca_vlc_grid_2;
VLC     ff_dca_vlc_grid_3;
VLC     ff_dca_vlc_rsd;

av_cold void ff_dca_init_vlcs(void)
{
    static VLC_TYPE dca_table[30214][2];
    static int vlcs_initialized = 0;
    int i, j, k = 0;

    if (vlcs_initialized)
        return;

#define DCA_INIT_VLC(vlc, a, b, c, d)                                       \
    do {                                                                    \
        vlc.table           = &dca_table[vlc_offs[k]];                      \
        vlc.table_allocated = vlc_offs[k + 1] - vlc_offs[k];                \
        init_vlc(&vlc, a, b, c, 1, 1, d, 2, 2, INIT_VLC_USE_NEW_STATIC);    \
        k++;                                                                \
    } while (0)

    ff_dca_vlc_bit_allocation.offset    = 1;
    ff_dca_vlc_bit_allocation.max_depth = 2;
    for (i = 0; i < 5; i++)
        DCA_INIT_VLC(ff_dca_vlc_bit_allocation.vlc[i], bitalloc_12_vlc_bits[i], 12,
                     bitalloc_12_bits[i], bitalloc_12_codes[i]);

    ff_dca_vlc_scale_factor.offset    = -64;
    ff_dca_vlc_scale_factor.max_depth = 2;
    for (i = 0; i < 5; i++)
        DCA_INIT_VLC(ff_dca_vlc_scale_factor.vlc[i], SCALES_VLC_BITS, 129,
                     scales_bits[i], scales_codes[i]);

    ff_dca_vlc_transition_mode.offset    = 0;
    ff_dca_vlc_transition_mode.max_depth = 1;
    for (i = 0; i < 4; i++)
        DCA_INIT_VLC(ff_dca_vlc_transition_mode.vlc[i], tmode_vlc_bits[i], 4,
                     tmode_bits[i], tmode_codes[i]);

    for (i = 0; i < DCA_CODE_BOOKS; i++) {
        ff_dca_vlc_quant_index[i].offset    = bitalloc_offsets[i];
        ff_dca_vlc_quant_index[i].max_depth = 1 + (i > 4);
        for (j = 0; bitalloc_codes[i][j]; j++)
            DCA_INIT_VLC(ff_dca_vlc_quant_index[i].vlc[j], bitalloc_maxbits[i][j],
                         bitalloc_sizes[i], bitalloc_bits[i][j], bitalloc_codes[i][j]);
    }

#define LBR_INIT_VLC(vlc, tab, nb_bits)                                 \
    do {                                                                \
        vlc.table           = &dca_table[vlc_offs[k]];                  \
        vlc.table_allocated = vlc_offs[k + 1] - vlc_offs[k];            \
        ff_init_vlc_sparse(&vlc, nb_bits, FF_ARRAY_ELEMS(tab##_codes),  \
                           &tab##_bitvals[0], 2, 1,                     \
                           tab##_codes, 2, 2,                           \
                           &tab##_bitvals[1], 2, 1,                     \
                           INIT_VLC_LE | INIT_VLC_USE_NEW_STATIC);      \
        k++;                                                            \
    } while (0)

    LBR_INIT_VLC(ff_dca_vlc_tnl_grp[0],  tnl_grp_0,   9);
    LBR_INIT_VLC(ff_dca_vlc_tnl_grp[1],  tnl_grp_1,   9);
    LBR_INIT_VLC(ff_dca_vlc_tnl_grp[2],  tnl_grp_2,   9);
    LBR_INIT_VLC(ff_dca_vlc_tnl_grp[3],  tnl_grp_3,   9);
    LBR_INIT_VLC(ff_dca_vlc_tnl_grp[4],  tnl_grp_4,   9);
    LBR_INIT_VLC(ff_dca_vlc_tnl_scf,     tnl_scf,     9);
    LBR_INIT_VLC(ff_dca_vlc_damp,        damp,        6);
    LBR_INIT_VLC(ff_dca_vlc_dph,         dph,         6);
    LBR_INIT_VLC(ff_dca_vlc_fst_rsd_amp, fst_rsd_amp, 9);
    LBR_INIT_VLC(ff_dca_vlc_rsd_apprx,   rsd_apprx,   5);
    LBR_INIT_VLC(ff_dca_vlc_rsd_amp,     rsd_amp,     9);
    LBR_INIT_VLC(ff_dca_vlc_avg_g3,      avg_g3,      9);
    LBR_INIT_VLC(ff_dca_vlc_st_grid,     st_grid,     9);
    LBR_INIT_VLC(ff_dca_vlc_grid_2,      grid_2,      9);
    LBR_INIT_VLC(ff_dca_vlc_grid_3,      grid_3,      9);
    LBR_INIT_VLC(ff_dca_vlc_rsd,         rsd,         6);

    vlcs_initialized = 1;
}

uint32_t ff_dca_vlc_calc_quant_bits(int *values, uint8_t n, uint8_t sel, uint8_t table)
{
    uint8_t i, id;
    uint32_t sum = 0;
    for (i = 0; i < n; i++) {
        id = values[i] - bitalloc_offsets[table];
        av_assert0(id < bitalloc_sizes[table]);
        sum += bitalloc_bits[table][sel][id];
    }
    return sum;
}

void ff_dca_vlc_enc_quant(PutBitContext *pb, int *values, uint8_t n, uint8_t sel, uint8_t table)
{
    uint8_t i, id;
    for (i = 0; i < n; i++) {
        id = values[i] - bitalloc_offsets[table];
        av_assert0(id < bitalloc_sizes[table]);
        put_bits(pb, bitalloc_bits[table][sel][id], bitalloc_codes[table][sel][id]);
    }
}
